-- CONFIG
local SCALE = 1.05
local TEXTURE = "Interface\\Addons\\vTooltip\\MinimalistHalf"
local POSITION = true -- Set to true to set tooltip position at bottom right of screen
local POS_X = -80      -- Horizontal offset (Negative moves left)
local POS_Y = 80       -- Vertical offset (Positive moves up)

------------------------------------------------------------------
-- HELPERS
------------------------------------------------------------------

-- Apply Border Color to edges/corners ONLY
local function ApplyBorderColor(self, r, g, b)
    if not self.NineSlice then return end
    local nr, ng, nb = r or 0.1, g or 0.1, b or 0.1
    local regions = {
        "TopEdge", "BottomEdge", "LeftEdge", "RightEdge",
        "TopLeftCorner", "TopRightCorner", "BottomLeftCorner", "BottomRightCorner"
    }
    for _, region in ipairs(regions) do
        if self.NineSlice[region] then
            self.NineSlice[region]:SetVertexColor(nr, ng, nb, 1)
        end
    end
end

-- Get intended color for a unit
local function GetUnitColor(unit)
    if UnitIsPlayer(unit) then
        local _, class = UnitClass(unit)
        local color = RAID_CLASS_COLORS[class]
        if color then return color.r, color.g, color.b end
    else
        local reaction = UnitReaction(unit, "player")
        if reaction then
            local color = FACTION_BAR_COLORS[reaction]
            if color then return color.r, color.g, color.b end
        end
    end
    return 0.5, 0.5, 0.5 -- Fallback grey
end

------------------------------------------------------------------
-- CORE TOOLTIP LOGIC
------------------------------------------------------------------

GameTooltip:SetScale(SCALE)

-- 1. ITEM BORDER COLORING
local function OnTooltipSetItem(self)
    local _, link = self:GetItem()
    if link then
        local quality = select(3, GetItemInfo(link))
        if quality and quality > 1 then
            local r, g, b = GetItemQualityColor(quality)
            ApplyBorderColor(self, r, g, b)
            return
        end
    end
    ApplyBorderColor(self)
end

-- 2. FORCED NAME COLORING
hooksecurefunc(GameTooltipTextLeft1, "SetTextColor", function(self, r, g, b)
    local _, unit = GameTooltip:GetUnit()
    if unit and UnitIsPlayer(unit) then
        local ur, ug, ub = GetUnitColor(unit)
        if (r ~= ur or g ~= ug or b ~= ub) then
            self:SetTextColor(ur, ug, ub)
        end
    end
end)

-- 3. FORCED HEALTH BAR COLORING
hooksecurefunc(GameTooltipStatusBar, "SetStatusBarColor", function(self, r, g, b)
    local _, unit = GameTooltip:GetUnit()
    if unit then
        local ur, ug, ub = GetUnitColor(unit)
        if (r ~= ur or g ~= ug or b ~= ub) then
            self:SetStatusBarColor(ur, ug, ub)
        end
    end
end)

-- 4. UNIT HOOK
GameTooltip:HookScript("OnTooltipSetUnit", function(self)
    local _, unit = self:GetUnit()
    if not unit then return end

    -- Strip PvP Titles & Color Name
    if UnitIsPlayer(unit) then
        local name = UnitName(unit)
        if name then
            GameTooltipTextLeft1:SetText(name)
            local r, g, b = GetUnitColor(unit)
            GameTooltipTextLeft1:SetTextColor(r, g, b)
        end
    end

    local guildName, guildRank = GetGuildInfo(unit)
    local level = UnitLevel(unit)
    local race = UnitRace(unit)
    local isPlayer = UnitIsPlayer(unit)
    
    for i = 2, self:NumLines() do
        local line = _G["GameTooltipTextLeft"..i]
        if line then
            local text = line:GetText()
            if text then
                -- A. HIJACK BLIZZARD GUILD LINE
                -- Detects the line Blizzard added and replaces it with our styled version
                if guildName and text:find(guildName) then
                    line:SetText("<" .. guildName .. "> " .. guildRank)
                    line:SetTextColor(1, 0.3, 1) -- Pink color
                
                -- B. MODIFY LEVEL LINE
                elseif text:find(LEVEL) or (race and text:find(race)) then
                    -- Strip "(Player)" text
                    line:SetText(text:gsub("%s?%(?[Pp]layer%)?", ""))
                    
                    -- Apply Difficulty Coloring (NPCs only)
                    if not isPlayer then
                        if level and level > 0 then
                            local color = GetQuestDifficultyColor(level)
                            line:SetTextColor(color.r, color.g, color.b)
                        elseif level == -1 then
                            line:SetTextColor(1, 0, 0) -- Boss/Skull Red
                        end
                    else
                        line:SetTextColor(1, 1, 1) -- Players stay white
                    end
                end
            end
        end
    end

    -- Target of Target
    local targetUnit = unit .. "target"
    if UnitExists(targetUnit) then
        local targetName = UnitName(targetUnit)
        local tr, tg, tb = GetUnitColor(targetUnit)
        self:AddDoubleLine("Targeting:", targetName, 0.7, 0.7, 0.7, tr, tg, tb)
    end

    -- Health Bar Setup
    if GameTooltipStatusBar then
        GameTooltipStatusBar:SetStatusBarTexture(TEXTURE)
        local r, g, b = GetUnitColor(unit)
        GameTooltipStatusBar:SetStatusBarColor(r, g, b)
    end
end)

------------------------------------------------------------------
-- POSITIONING
------------------------------------------------------------------

if POSITION then
    hooksecurefunc("GameTooltip_SetDefaultAnchor", function(tooltip, parent)
        tooltip:SetOwner(parent, "ANCHOR_NONE")
        tooltip:ClearAllPoints()
        tooltip:SetPoint("BOTTOMRIGHT", UIParent, "BOTTOMRIGHT", POS_X, POS_Y)
    end)
end

-- Health Bar inside Top
GameTooltipStatusBar:ClearAllPoints()
GameTooltipStatusBar:SetPoint("TOPLEFT", GameTooltip, "TOPLEFT", 3, -3)
GameTooltipStatusBar:SetPoint("TOPRIGHT", GameTooltip, "TOPRIGHT", -3, -3)
GameTooltipStatusBar:SetHeight(8)

hooksecurefunc(GameTooltipStatusBar, "SetPoint", function(self, point, anchor)
    if anchor ~= GameTooltip then
        self:ClearAllPoints()
        self:SetPoint("TOPLEFT", GameTooltip, "TOPLEFT", 3, -3)
        self:SetPoint("TOPRIGHT", GameTooltip, "TOPRIGHT", -3, -3)
    end
end)

------------------------------------------------------------------
-- UNIVERSAL HOOKS
------------------------------------------------------------------

local tooltips = {
    GameTooltip, ItemRefTooltip, ShoppingTooltip1, ShoppingTooltip2, 
    ShoppingTooltip3, WorldMapTooltip,
}

for _, tt in pairs(tooltips) do
    if tt then
        tt:HookScript("OnTooltipSetItem", OnTooltipSetItem)
        tt:HookScript("OnShow", function(self)
            if not self:GetItem() then
                ApplyBorderColor(self)
            end
        end)
    end
end